"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@nestjs/core");
const common_1 = require("@nestjs/common");
const swagger_1 = require("@nestjs/swagger");
const os_1 = require("os");
const app_module_1 = require("./app.module");
const logging_interceptor_1 = require("./common/interceptors/logging.interceptor");
const transform_interceptor_1 = require("./common/interceptors/transform.interceptor");
function getLocalIP() {
    const interfaces = (0, os_1.networkInterfaces)();
    for (const interfaceName of Object.keys(interfaces)) {
        const nets = interfaces[interfaceName];
        if (!nets)
            continue;
        for (const net of nets) {
            if (net.family === 'IPv4' && !net.internal) {
                if (net.address.startsWith('192.168.') ||
                    net.address.startsWith('10.0.') ||
                    net.address.startsWith('172.')) {
                    return net.address;
                }
            }
        }
    }
    return 'localhost';
}
async function bootstrap() {
    const app = await core_1.NestFactory.create(app_module_1.AppModule);
    app.enableCors({
        origin: true,
        methods: 'GET,HEAD,PUT,PATCH,POST,DELETE,OPTIONS',
        credentials: true,
    });
    app.useGlobalPipes(new common_1.ValidationPipe({
        whitelist: true,
        forbidNonWhitelisted: true,
        transform: true,
    }));
    app.useGlobalInterceptors(new logging_interceptor_1.LoggingInterceptor(), new transform_interceptor_1.TransformInterceptor());
    const config = new swagger_1.DocumentBuilder()
        .setTitle('E-commerce API')
        .setDescription('API documentation for e-commerce platform')
        .setVersion('1.0')
        .addBearerAuth()
        .addTag('Auth')
        .addTag('Products')
        .addTag('Admin')
        .addTag('Categories')
        .addTag('Customer')
        .addTag('Orders')
        .addTag('Payments')
        .addTag('Vendor')
        .addTag('Health')
        .build();
    const document = swagger_1.SwaggerModule.createDocument(app, config);
    swagger_1.SwaggerModule.setup('api-docs', app, document);
    const port = process.env.PORT || 5000;
    const localIP = getLocalIP();
    await app.listen(port, '0.0.0.0');
    console.log(`E-commerce API Server Running`);
    console.log(`Local: http://localhost:${port}`);
    console.log(`Network: http://${localIP}:${port}`);
    console.log(`API Docs: http://${localIP}:${port}/api-docs`);
    console.log(`Health: http://${localIP}:${port}/health`);
    console.log(`phpMyAdmin: http://localhost:8080`);
    console.log(`Database: mysql://localhost:3306/ecommerce_db`);
    console.log(``);
    console.log(`server url on local network:`);
    console.log(`   http://${localIP}:${port}`);
}
bootstrap();
//# sourceMappingURL=main.js.map